<?php
session_start();
require_once 'config.php'; 
require_once 'includes/emission_factors_config.php';

// Include PhpSpreadsheet autoloader
require_once 'vendor/autoload.php'; // Adjust path if your vendor directory is elsewhere

use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\Style\Font;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use PhpOffice\PhpSpreadsheet\Style\Border;
use PhpOffice\PhpSpreadsheet\Style\Fill;

if (!isset($_SESSION['general_id'])) {
    header("Content-Type: text/plain");
    die("Error: No submission data found in session. Please generate a summary report first.");
}
$general_id = $_SESSION['general_id'];

// Initialize Arrays
$years = [2021, 2022, 2023, 2024];
$categories_summary = [ // USE THIS NAME CONSISTENTLY for summary calculations
    'total_scope1' => array_fill_keys($years, 0.0),
    'total_scope2' => array_fill_keys($years, 0.0),
    'total_scope1_plus_2' => array_fill_keys($years, 0.0),
    's1_stationary' => array_fill_keys($years, 0.0),
    's1_mobile' => array_fill_keys($years, 0.0),
    's1_fugitive' => array_fill_keys($years, 0.0),
    's2_electricity' => array_fill_keys($years, 0.0),
    's2_other_energy' => array_fill_keys($years, 0.0),
];

// Arrays to hold RAW input details
$general_info_data = null;
$raw_s1_stationary = []; 
$raw_s1_mobile = []; 
$raw_s1_fugitive = [];
$raw_s2_electricity = []; 
$raw_s2_shc = [];


// --- START: DATA FETCHING & AGGREGATION (Populate BOTH summary AND raw data arrays here) ---
// Fetch General Info (Only needed once)
$stmt_gi = mysqli_prepare($conn, "SELECT * FROM general_info WHERE id = ?");
if ($stmt_gi) {
    mysqli_stmt_bind_param($stmt_gi, "i", $general_id); mysqli_stmt_execute($stmt_gi);
    $result_gi = mysqli_stmt_get_result($stmt_gi);
    $general_info_data = mysqli_fetch_assoc($result_gi);
    mysqli_stmt_close($stmt_gi);
} // else { error_log("Error preparing general_info: " . mysqli_error($conn)); }

// Fetch Scope 1 Main ID
$scope1_emission_id = null;
$sql_s1_main = "SELECT id FROM scope1_emissions WHERE general_id = ? LIMIT 1";
$stmt_s1_main = mysqli_prepare($conn, $sql_s1_main);
if ($stmt_s1_main) {
    mysqli_stmt_bind_param($stmt_s1_main, "i", $general_id);
    if (mysqli_stmt_execute($stmt_s1_main)) {
        $res_s1_main = mysqli_stmt_get_result($stmt_s1_main);
        if ($res_s1_main) { 
			$s1_data = mysqli_fetch_assoc($res_s1_main);
			if ($s1_data) $scope1_emission_id = $s1_data['id'];
		}
    } 
	mysqli_stmt_close($stmt_s1_main);
} // else { error_log("Error preparing S1 main: " . mysqli_error($conn)); }

if ($scope1_emission_id) {
    // Stationary Combustion Details
    $stmt_stat = mysqli_prepare($conn, "SELECT * FROM stationary_combustion_details WHERE scope1_emission_id = ?"); // Select all for raw data
    if($stmt_stat) {
        mysqli_stmt_bind_param($stmt_stat, "i", $scope1_emission_id); mysqli_stmt_execute($stmt_stat);
        $result_stat = mysqli_stmt_get_result($stmt_stat);
        while ($row = mysqli_fetch_assoc($result_stat)) {
            $raw_s1_stationary[] = $row; // STORE RAW DATA
            $ef = $emission_factors_data['stationary'][$row['fuel_type']][$row['input_unit']] ?? 0;

            if ($ef > 0) { 
				foreach ($years as $year) {
					$qty_col = 'quantity_' . $year;
					$co2e_year = (floatval($row[$qty_col] ?? 0) * $ef); /* Adjust /1000 based on EF unit */
					$categories_summary['s1_stationary'][$year] += $co2e_year;
					$categories_summary['total_scope1'][$year] += $co2e_year; 
				}
			}
        } 
		mysqli_stmt_close($stmt_stat);
    } // else { error_log(...); }

    // Mobile Combustion Details
    $stmt_mob = mysqli_prepare($conn, "SELECT * FROM mobile_combustion_details WHERE scope1_emission_id = ?");
    if($stmt_mob){
        mysqli_stmt_bind_param($stmt_mob, "i", $scope1_emission_id); mysqli_stmt_execute($stmt_mob);
        $result_mob = mysqli_stmt_get_result($stmt_mob);
        while ($row = mysqli_fetch_assoc($result_mob)) {
            $raw_s1_mobile[] = $row; // STORE RAW DATA
            $ef = $emission_factors_data['mobile'][$row['fuel_type']][$row['input_unit']] ?? 0;
            if ($ef > 0) { 
				foreach ($years as $year) { 
					$qty_col = 'quantity_' . $year; 
					$co2e_year = (floatval($row[$qty_col] ?? 0) * $ef); /* Adjust /1000 */ 
					$categories_summary['s1_mobile'][$year] += $co2e_year; 
					$categories_summary['total_scope1'][$year] += $co2e_year; 
				} 
			}
        } 
		mysqli_stmt_close($stmt_mob);
    } // else { error_log(...); }

    // Fugitive Emissions Details
    $stmt_fug = mysqli_prepare($conn, "SELECT * FROM fugitive_emission_details WHERE scope1_emission_id = ?");
    if($stmt_fug){
        mysqli_stmt_bind_param($stmt_fug, "i", $scope1_emission_id); mysqli_stmt_execute($stmt_fug);
        $result_fug = mysqli_stmt_get_result($stmt_fug);
        while ($row = mysqli_fetch_assoc($result_fug)) {
            $raw_s1_fugitive[] = $row; // STORE RAW DATA
            $gwp = $emission_factors_data['fugitive'][$row['gas_type']][$row['input_unit']] ?? $emission_factors_data['fugitive'][$row['gas_type']]['kg'] ?? 0;
            if ($gwp > 0) { 
				foreach ($years as $year) { 
					$qty_col = 'quantity_' . $year; 
					$qty_original_unit = floatval($row[$qty_col] ?? 0); 
					$qty_in_tonnes = ($row['input_unit'] === 'kg') ? $qty_original_unit : (($row['input_unit'] === 'metric tonnes') ? $qty_original_unit : 0) ; 
					if ($qty_in_tonnes > 0) { 
						$co2e_year = $qty_in_tonnes * $gwp; 
						$categories_summary['s1_fugitive'][$year] += $co2e_year; 
						$categories_summary['total_scope1'][$year] += $co2e_year; 
					} 
				} 
			}
        } 
		mysqli_stmt_close($stmt_fug);
    } // else { error_log(...); }
}

// Fetch Scope 2 Main ID and include_shc flag
$scope2_emission_id = null;
$s2_main_db_data = null; 
$sql_s2_main = "SELECT id, include_steam_heat_cooling FROM scope2_emissions WHERE general_id = ? LIMIT 1";
$stmt_s2_main = mysqli_prepare($conn, $sql_s2_main);
if($stmt_s2_main) {
    mysqli_stmt_bind_param($stmt_s2_main, "i", $general_id);
    if(mysqli_stmt_execute($stmt_s2_main)) {
        $res_s2_main = mysqli_stmt_get_result($stmt_s2_main);
        if($res_s2_main) { $s2_main_db_data = mysqli_fetch_assoc($res_s2_main); if($s2_main_db_data) $scope2_emission_id = $s2_main_db_data['id']; }
    } 
	mysqli_stmt_close($stmt_s2_main);
} // else { error_log(...); }

if ($scope2_emission_id) {
    // Electricity Consumption Details
    $stmt_elec = mysqli_prepare($conn, "SELECT * FROM electricity_consumption_details WHERE scope2_emission_id = ?");
    if($stmt_elec){
        mysqli_stmt_bind_param($stmt_elec, "i", $scope2_emission_id); 
		mysqli_stmt_execute($stmt_elec);
        $result_elec = mysqli_stmt_get_result($stmt_elec);
        while ($row = mysqli_fetch_assoc($result_elec)) {
            $raw_s2_electricity[] = $row; // STORE RAW DATA
			
			// Get region and input_unit FROM THE CURRENT $row
            $current_row_region = $row['region'];
            $current_row_input_unit = $row['input_unit'];
			$year_column_keys = ['col1', 'col2', 'col3', 'col4'];

            // Iterate through each year to apply the year-specific EF
            foreach (array_keys($years) as $year_index) { // $year_index will be 0, 1, 2, 3
                $current_year = $years[$year_index]; // Get 2021, 2022, etc.
                $ef_key_for_year = $year_column_keys[$year_index]; // Get col1, col2, etc.

                $qty_col = 'quantity_' . $current_year;
                $quantity_value = floatval($row[$qty_col] ?? 0);

                if ($quantity_value > 0) {
                    $ef_for_year = 0;
                    if (isset($emission_factors_data_scope2['electricity'][$current_row_region]) &&
                        isset($emission_factors_data_scope2['electricity'][$current_row_region][$current_row_input_unit]) &&
                        isset($emission_factors_data_scope2['electricity'][$current_row_region][$current_row_input_unit][$ef_key_for_year])) {
                        
                        $ef_for_year = $emission_factors_data_scope2['electricity'][$current_row_region][$current_row_input_unit][$ef_key_for_year];
                    } // else { error_log(...); }
                    
                    if ($ef_for_year > 0) {
                        $co2e_year = ($quantity_value * $ef_for_year);
                        // Use $current_year to store in $categories_summary
                        $categories_summary['s2_electricity'][$current_year] += $co2e_year;
                        $categories_summary['total_scope2'][$current_year] += $co2e_year;
                    }
                }
            }
		} 
		mysqli_stmt_close($stmt_elec);
    } else { 
        // error_log("Error preparing S2 Electricity details: ".mysqli_error($conn)); 
    }

    // Steam, Heat & Cooling Details
    if ($s2_main_db_data && $s2_main_db_data['include_steam_heat_cooling']) {
        $stmt_shc = mysqli_prepare($conn, "SELECT * FROM steam_heat_cooling_details WHERE scope2_emission_id = ?");
        if($stmt_shc){
            mysqli_stmt_bind_param($stmt_shc, "i", $scope2_emission_id); 
			mysqli_stmt_execute($stmt_shc);
            $result_shc = mysqli_stmt_get_result($stmt_shc);
            while ($row = mysqli_fetch_assoc($result_shc)) {
                $raw_s2_shc[] = $row; // STORE RAW DATA
                $fixed_shc_unit = 'metric tonnes';
                $ef = $emission_factors_data_scope2['steam_heat_cooling'][$row['energy_type']][$fixed_shc_unit] ?? 0; // Used corrected structure
                if ($ef > 0) { 
					foreach ($years as $year) { 
						$qty_col = 'quantity_' . $year; $co2e_year = (floatval($row[$qty_col] ?? 0) * $ef); /* Adjust /1000 */ 
						$categories_summary['s2_other_energy'][$year] += $co2e_year; $categories_summary['total_scope2'][$year] += $co2e_year; 
					} 
				}
            } 
			mysqli_stmt_close($stmt_shc);
        } // else { error_log(...); }
    }
}

// --- NEW: Fetch Annual Revenue Data ---
$annual_revenues_by_year = array_fill_keys($years, null);
$stmt_revenue = mysqli_prepare($conn, "SELECT revenue_2021, revenue_2022, revenue_2023, revenue_2024 FROM annual_revenues WHERE general_id = ? LIMIT 1");
if ($stmt_revenue) {
    mysqli_stmt_bind_param($stmt_revenue, "i", $general_id);
    if (mysqli_stmt_execute($stmt_revenue)) {
        $result_revenue = mysqli_stmt_get_result($stmt_revenue);
        if ($revenue_data_row = mysqli_fetch_assoc($result_revenue)) {
            foreach ($years as $year) {
                $annual_revenues_by_year[$year] = isset($revenue_data_row['revenue_'.$year]) ? floatval($revenue_data_row['revenue_'.$year]) : null;
            }
        }
    } mysqli_stmt_close($stmt_revenue);
}

// --- NEW: Initialize and Calculate Intensity Metrics ---
$intensity_metrics = [
    'scope1_intensity' => array_fill_keys($years, null),
    'scope2_intensity' => array_fill_keys($years, null),
    'total_intensity' => array_fill_keys($years, null),
];
// --- NEW: Fetch Annual Revenue Data ---

// Calculate Total Scope 1 + 2
foreach ($years as $year) {
	$categories_summary['total_scope1_plus_2'][$year] = ($categories_summary['total_scope1'][$year] ?? 0.0) + ($categories_summary['total_scope2'][$year] ?? 0.0);
	
    $total_s1 = $categories_summary['total_scope1'][$year] ?? 0.0;
    $total_s2 = $categories_summary['total_scope2'][$year] ?? 0.0;
    $revenue_for_this_year = $annual_revenues_by_year[$year] ?? null;

    if ($revenue_for_this_year !== null && $revenue_for_this_year > 0) {
        $divisor_revenue = $revenue_for_this_year / 1000000.0; // For "metric tonnes CO2e/million RM revenue"
        if ($divisor_revenue > 0) { 
             $intensity_metrics['scope1_intensity'][$year] = $total_s1 / $divisor_revenue;
             $intensity_metrics['scope2_intensity'][$year] = $total_s2 / $divisor_revenue;
             $intensity_metrics['total_intensity'][$year] = ($total_s1 + $total_s2) / $divisor_revenue;
        }
    }
}
// --- END: DATA FETCHING & AGGREGATION LOGIC ---


// --- Create new Spreadsheet object ---
$spreadsheet = new Spreadsheet();
$sheet = $spreadsheet->getActiveSheet();
$sheet->setTitle('GHG Summary Report');

// --- Styles ---
$headerStyle = [
    'font' => ['bold' => true, 'size' => 11],
    'alignment' => ['horizontal' => Alignment::HORIZONTAL_CENTER, 'vertical' => Alignment::VERTICAL_CENTER],
    'fill' => ['fillType' => Fill::FILL_SOLID, 'startColor' => ['argb' => 'FFD9D9D9']]
];
$sectionTitleStyle = [
    'font' => ['bold' => true, 'size' => 14],
    'alignment' => ['vertical' => Alignment::VERTICAL_CENTER]
];
$subSectionTitleStyle = [
    'font' => ['bold' => true, 'size' => 12, 'italic' => true],
    'alignment' => ['vertical' => Alignment::VERTICAL_CENTER]
];
$boldStyle = ['font' => ['bold' => true]];
$rightAlignStyle = ['alignment' => ['horizontal' => Alignment::HORIZONTAL_RIGHT]];

// --- Start writing data ---
$rowIndex = 1;

// Report Title
$sheet->mergeCells('A'.$rowIndex.':F'.$rowIndex);
$sheet->setCellValue('A'.$rowIndex, 'GHG Emissions Summary Report');
$sheet->getStyle('A'.$rowIndex)->applyFromArray($sectionTitleStyle)->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);
$rowIndex++;
$sheet->setCellValue('A'.$rowIndex, 'Report for General ID:'); $sheet->setCellValue('B'.$rowIndex, $general_id);
$sheet->setCellValue('D'.$rowIndex, 'Generated:'); $sheet->setCellValue('E'.$rowIndex, date("Y-m-d H:i:s"));
$rowIndex += 2;

// --- SECTION 1: SUMMARY TABLE ---
$sheet->setCellValue('A'.$rowIndex, 'Total GHG Emissions')->getStyle('A'.$rowIndex)->applyFromArray($subSectionTitleStyle);
$rowIndex++;

// Column Headers for the data rows (Category, Unit, Year1, Year2, Year3, Year4)
$dataColumnHeaders = ['Category', 'Unit'];
foreach($years as $year) {
    $dataColumnHeaders[] = $year;
}
$sheet->fromArray($dataColumnHeaders, NULL, 'A'.$rowIndex);
$sheet->getStyle('A'.$rowIndex.':'. $sheet->getHighestDataColumn($rowIndex).$rowIndex)->applyFromArray($headerStyle);
$rowIndex++;

function write_summary_row_xlsx($sheet, &$rowIndex, $label, $unit, $data_array, $year_keys, $isBold = false, $numberFormat = '#,##0.00') { // Default to 2DP if that's what you mostly want
    $col_letter = 'A';
    $sheet->setCellValue($col_letter++.$rowIndex, $label);
    $sheet->setCellValue($col_letter++.$rowIndex, $unit);
    foreach ($year_keys as $year) {
        $cellValue = floatval($data_array[$year] ?? 0.0); // Ensure it's a float

        // Write the raw numeric value to the cell
        $sheet->setCellValue($col_letter.$rowIndex, $cellValue); 
        
        // Apply Excel's number format to the cell
        $sheet->getStyle($col_letter.$rowIndex)->getNumberFormat()->setFormatCode($numberFormat); 
        $sheet->getStyle($col_letter.$rowIndex)->getAlignment()->setHorizontal(Alignment::HORIZONTAL_RIGHT);
        $col_letter++;
    }
    if ($isBold) {
        $endColLetter = chr(ord('A') + 1 + count($year_keys) -1);
        $sheet->getStyle('A'.$rowIndex.':'.$endColLetter.$rowIndex)->getFont()->setBold(true);
    }
    $rowIndex++;
}

$numberFormat4dp = '#,##0.0000';
$numberFormat2dp = '#,##0.00';

write_summary_row_xlsx($sheet, $rowIndex, 'Total Scope 1', 'metric tonnes of CO2e', $categories_summary['total_scope1'], $years, false, $numberFormat2dp);
write_summary_row_xlsx($sheet, $rowIndex, 'Total Scope 2', 'metric tonnes of CO2e', $categories_summary['total_scope2'], $years, false, $numberFormat2dp);
write_summary_row_xlsx($sheet, $rowIndex, 'Total Scope 1+2', 'metric tonnes of CO2e', $categories_summary['total_scope1_plus_2'], $years, true, $numberFormat2dp);
$rowIndex++;

// --- SECTION 2: INTENSITY METRICS ---
$sheet->setCellValue('A'.$rowIndex, 'Total Intensity')->getStyle('A'.$rowIndex)->applyFromArray($subSectionTitleStyle); // Use main section title style
$rowIndex++;

// Headers for intensity (can reuse $dataColumnHeaders)
$sheet->fromArray($dataColumnHeaders, NULL, 'A'.$rowIndex);
$sheet->getStyle('A'.$rowIndex.':F'.$rowIndex)->applyFromArray($headerStyle);
$rowIndex++;

$intensityUnit = 'metric tonnes CO2e/million RM revenue'; // Adjust if your divisor for revenue is different
$numberFormatIntensity = '#,##0.00'; // For intensity values

write_summary_row_xlsx($sheet, $rowIndex, 'Scope 1 Intensity', $intensityUnit, $intensity_metrics['scope1_intensity'], $years, false, $numberFormatIntensity);
write_summary_row_xlsx($sheet, $rowIndex, 'Scope 2 Intensity', $intensityUnit, $intensity_metrics['scope2_intensity'], $years, false, $numberFormatIntensity);
write_summary_row_xlsx($sheet, $rowIndex, 'Total Intensity (Scope 1+2)', $intensityUnit, $intensity_metrics['total_intensity'], $years, true, $numberFormatIntensity);
$rowIndex++;

$sheet->setCellValue('A'.$rowIndex, 'Scope 1 Breakdown')->getStyle('A'.$rowIndex)->applyFromArray($subSectionTitleStyle); $rowIndex++;
write_summary_row_xlsx($sheet, $rowIndex, 'Stationary Combustion', 'metric tonnes of CO2e', $categories_summary['s1_stationary'], $years, false, $numberFormat2dp);
write_summary_row_xlsx($sheet, $rowIndex, 'Mobile Combustion', 'metric tonnes of CO2e', $categories_summary['s1_mobile'], $years, false, $numberFormat2dp);
write_summary_row_xlsx($sheet, $rowIndex, 'Fugitive Emissions (Refrigerants)', 'metric tonnes of CO2e', $categories_summary['s1_fugitive'], $years, false, $numberFormat2dp);
$rowIndex++;

$sheet->setCellValue('A'.$rowIndex, 'Scope 2 Breakdown')->getStyle('A'.$rowIndex)->applyFromArray($subSectionTitleStyle); $rowIndex++;
write_summary_row_xlsx($sheet, $rowIndex, 'Electricity Consumption', 'metric tonnes of CO2e', $categories_summary['s2_electricity'], $years, false, $numberFormat2dp);
write_summary_row_xlsx($sheet, $rowIndex, 'Other Energy Consumption (Steam, Heat, Cooling)', 'metric tonnes of CO2e', $categories_summary['s2_other_energy'], $years, false, $numberFormat2dp);

// --- SECTION 2: GENERAL INFORMATION ---
$rowIndex += 2;
$sheet->setCellValue('A'.$rowIndex, 'General Information')->getStyle('A'.$rowIndex)->applyFromArray($subSectionTitleStyle); $rowIndex++;
if ($general_info_data) {
    foreach ($general_info_data as $key => $value) {
        $sheet->setCellValue('A'.$rowIndex, ucwords(str_replace('_', ' ', $key)));
        $sheet->setCellValue('B'.$rowIndex, $value);
        $rowIndex++;
    }
} else { $sheet->setCellValue('A'.$rowIndex, 'No general information found.'); $rowIndex++; }

// --- SECTION 3: SCOPE 1 INPUT DETAILS ---
$rowIndex += 2;
$sheet->setCellValue('A'.$rowIndex, 'Scope 1 Input Details')->getStyle('A'.$rowIndex)->applyFromArray($subSectionTitleStyle); $rowIndex++;

// Stationary Inputs
$sheet->setCellValue('A'.$rowIndex, 'Stationary Combustion Inputs')->getStyle('A'.$rowIndex)->getFont()->setItalic(true)->setBold(true); $rowIndex++;
$s1StatHeader = ['Fuel Type', 'Input Unit', '2021', '2022', '2023', '2024'];
$sheet->fromArray($s1StatHeader, NULL, 'A'.$rowIndex);
$sheet->getStyle('A'.$rowIndex.':F'.$rowIndex)->applyFromArray($headerStyle); $rowIndex++;
if (!empty($raw_s1_stationary)) {
    foreach ($raw_s1_stationary as $item) {
        $sheet->fromArray([$item['fuel_type'], $item['input_unit'], $item['quantity_2021'], $item['quantity_2022'], $item['quantity_2023'], $item['quantity_2024']], NULL, 'A'.$rowIndex);
        $sheet->getStyle('C'.$rowIndex.':F'.$rowIndex)->getNumberFormat()->setFormatCode('#,##0.00');
        $rowIndex++;
    }
} else { $sheet->setCellValue('A'.$rowIndex, 'No stationary combustion input data.'); $rowIndex++; }
$rowIndex++;

// Mobile Inputs
$sheet->setCellValue('A'.$rowIndex, 'Mobile Combustion Inputs')->getStyle('A'.$rowIndex)->getFont()->setItalic(true)->setBold(true); $rowIndex++;
$s1MobHeader = ['Fuel Type', 'Input Unit', '2021', '2022', '2023', '2024'];
$sheet->fromArray($s1MobHeader, NULL, 'A'.$rowIndex);
$sheet->getStyle('A'.$rowIndex.':F'.$rowIndex)->applyFromArray($headerStyle); $rowIndex++;
if (!empty($raw_s1_mobile)) {
    foreach ($raw_s1_mobile as $item) {
        $sheet->fromArray([$item['fuel_type'], $item['input_unit'], $item['quantity_2021'], $item['quantity_2022'], $item['quantity_2023'], $item['quantity_2024']], NULL, 'A'.$rowIndex);
        $sheet->getStyle('C'.$rowIndex.':F'.$rowIndex)->getNumberFormat()->setFormatCode('#,##0.00');
        $rowIndex++;
    }
} else { $sheet->setCellValue('A'.$rowIndex, 'No mobile combustion input data.'); $rowIndex++; }
$rowIndex++;

// Fugitive Inputs
$sheet->setCellValue('A'.$rowIndex, 'Fugitive Emissions Inputs')->getStyle('A'.$rowIndex)->getFont()->setItalic(true)->setBold(true); $rowIndex++;
$s1FugHeader = ['Gas Type', 'Input Unit', '2021', '2022', '2023', '2024'];
$sheet->fromArray($s1FugHeader, NULL, 'A'.$rowIndex);
$sheet->getStyle('A'.$rowIndex.':F'.$rowIndex)->applyFromArray($headerStyle); $rowIndex++;
if (!empty($raw_s1_fugitive)) {
    foreach ($raw_s1_fugitive as $item) {
        $sheet->fromArray([$item['gas_type'], $item['input_unit'], $item['quantity_2021'], $item['quantity_2022'], $item['quantity_2023'], $item['quantity_2024']], NULL, 'A'.$rowIndex);
        $sheet->getStyle('C'.$rowIndex.':F'.$rowIndex)->getNumberFormat()->setFormatCode('#,##0.00');
        $rowIndex++;
    }
} else { $sheet->setCellValue('A'.$rowIndex, 'No fugitive emissions input data.'); $rowIndex++; }
$rowIndex++;


// --- SECTION 4: SCOPE 2 INPUT DETAILS ---
$sheet->setCellValue('A'.$rowIndex, 'Scope 2 Input Details')->getStyle('A'.$rowIndex)->applyFromArray($subSectionTitleStyle); $rowIndex++;

// Electricity Inputs
$sheet->setCellValue('A'.$rowIndex, 'Electricity Consumption Inputs')->getStyle('A'.$rowIndex)->getFont()->setItalic(true)->setBold(true); $rowIndex++;
$s2ElecHeader = ['Region', 'Input Unit', '2021', '2022', '2023', '2024'];
$sheet->fromArray($s2ElecHeader, NULL, 'A'.$rowIndex);
$sheet->getStyle('A'.$rowIndex.':F'.$rowIndex)->applyFromArray($headerStyle); $rowIndex++;
if (!empty($raw_s2_electricity)) {
    foreach ($raw_s2_electricity as $item) {
        $sheet->fromArray([$item['region'], $item['input_unit'], $item['quantity_2021'], $item['quantity_2022'], $item['quantity_2023'], $item['quantity_2024']], NULL, 'A'.$rowIndex);
        $sheet->getStyle('C'.$rowIndex.':F'.$rowIndex)->getNumberFormat()->setFormatCode('#,##0.00');
        $rowIndex++;
    }
} else { $sheet->setCellValue('A'.$rowIndex, 'No electricity input data.'); $rowIndex++; }
$rowIndex++;

// Steam, Heat & Cooling Inputs
if ($s2_main_db_data && $s2_main_db_data['include_steam_heat_cooling']) {
    $sheet->setCellValue('A'.$rowIndex, 'Steam, Heat & Cooling Inputs')->getStyle('A'.$rowIndex)->getFont()->setItalic(true)->setBold(true); $rowIndex++;
    $s2ShcHeader = ['Energy Type', 'Input Unit', '2021', '2022', '2023', '2024'];
    $sheet->fromArray($s2ShcHeader, NULL, 'A'.$rowIndex);
    $sheet->getStyle('A'.$rowIndex.':F'.$rowIndex)->applyFromArray($headerStyle); $rowIndex++;
    if (!empty($raw_s2_shc)) {
        foreach ($raw_s2_shc as $item) {
            $sheet->fromArray([$item['energy_type'], 'metric tonnes', $item['quantity_2021'], $item['quantity_2022'], $item['quantity_2023'], $item['quantity_2024']], NULL, 'A'.$rowIndex);
            $sheet->getStyle('C'.$rowIndex.':F'.$rowIndex)->getNumberFormat()->setFormatCode('#,##0.00');
            $rowIndex++;
        }
    } else { $sheet->setCellValue('A'.$rowIndex, 'No S/H/C input data.'); $rowIndex++; }
} else {
    //$sheet->setCellValue('A'.$rowIndex, 'Steam, Heat & Cooling section was not included.'); $rowIndex++;
}

// --- ANNUAL REVENUE INPUTS ---
$sheet->setCellValue('A'.$rowIndex, 'Revenue Inputs (Used for Intensity Calculation)')->getStyle('A'.$rowIndex)->applyFromArray($subSectionTitleStyle);
$rowIndex++;
$revenueHeader = ['Year', 'Revenue (RM)']; // Simple 2-column for this section
$sheet->fromArray($revenueHeader, NULL, 'A'.$rowIndex);
$sheet->getStyle('A'.$rowIndex.':B'.$rowIndex)->applyFromArray($headerStyle);
$rowIndex++;

if (!empty($annual_revenues_by_year) && count(array_filter($annual_revenues_by_year)) > 0) { // Check if there's any non-null revenue
    foreach ($years as $year) {
        $sheet->setCellValue('A'.$rowIndex, $year);
        $revenue_value = $annual_revenues_by_year[$year] ?? null;
        if ($revenue_value !== null) {
            $sheet->setCellValue('B'.$rowIndex, floatval($revenue_value));
            $sheet->getStyle('B'.$rowIndex)->getNumberFormat()->setFormatCode($numberFormat2dp); // Format revenue to 2dp
        } else {
            $sheet->setCellValue('B'.$rowIndex, '-'); // Or 'N/A'
        }
        $sheet->getStyle('B'.$rowIndex)->getAlignment()->setHorizontal(Alignment::HORIZONTAL_RIGHT);
        $rowIndex++;
    }
} else {
    $sheet->setCellValue('A'.$rowIndex, 'No annual revenue data submitted or all revenues are zero.');
    $rowIndex++;
}
$rowIndex++; // Blank row after this section

// --- SET COLUMN WIDTHS ---
$sheet->getColumnDimension('A')->setWidth(45); // Category / Field Name
$sheet->getColumnDimension('B')->setWidth(35); // Unit / Value
$sheet->getColumnDimension('C')->setWidth(15); // 2021
$sheet->getColumnDimension('D')->setWidth(15); // 2022
$sheet->getColumnDimension('E')->setWidth(15); // 2023
$sheet->getColumnDimension('F')->setWidth(15); // 2024

// --- Output the Excel file ---
$filename_xlsx = "ghg_full_report_general_id_" . $general_id . "_" . date('Ymd') . ".xlsx";
header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
header('Content-Disposition: attachment;filename="' . $filename_xlsx . '"');
header('Cache-Control: max-age=0');
header('Cache-Control: max-age=1'); // Not strictly necessary with other headers
header ('Expires: Mon, 26 Jul 1997 05:00:00 GMT'); // Date in the past
header ('Last-Modified: '.gmdate('D, d M Y H:i:s').' GMT'); // Always modified
header ('Cache-Control: cache, must-revalidate'); // HTTP/1.1
header ('Pragma: public'); // HTTP/1.0

$writer = new Xlsx($spreadsheet);
try {
    $writer->save('php://output');
} catch(\PhpOffice\PhpSpreadsheet\Writer\Exception $e) {
    // Log this error properly on the server
    error_log("PhpSpreadsheet Writer Exception: " . $e->getMessage());
    die("Error creating Excel file. Please try again later.");
}
exit;
?>